// -*- c++ -*-
/*=========================================================================

  Program:   Visualization Toolkit
  Module:    $RCSfile: vtkNetCDFReader.h,v $

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

/*-------------------------------------------------------------------------
  Copyright 2009 Sandia Corporation.
  Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
  the U.S. Government retains certain rights in this software.
-------------------------------------------------------------------------*/

// .NAME vtkFileSeriesWriter - meta-writer to write a file series
//
// .SECTION Description:
//
// vtkFileSeriesWriter is a meta-writer that can work with various writers
// to write a file series.  To the pipeline, it looks like a writer that
// requests time.  It updates the pipeline with each time step and then
// requests the real reader to write that out.
//

#ifndef __vtkFileSeriesWriter_h
#define __vtkFileSeriesWriter_h

#include "vtkWriter.h"

class VTK_EXPORT vtkFileSeriesWriter : public vtkWriter
{
 public:
  vtkTypeRevisionMacro(vtkFileSeriesWriter, vtkWriter);
  static vtkFileSeriesWriter *New();
  virtual void PrintSelf(ostream &os, vtkIndent indent);

  // Description:
  // The base filename.  If writing multiple time steps, an index is attached.
  vtkSetStringMacro(FileName);
  vtkGetStringMacro(FileName);

  // Description:
  // Set/get the internal writer.
  virtual void SetWriter(vtkWriter *);
  vtkGetObjectMacro(Writer, vtkWriter);

  // Description:
  // Name of the method used to set the file name of the internal
  // reader. By default, this is SetFileName.
  vtkSetStringMacro(FileNameMethod);
  vtkGetStringMacro(FileNameMethod);

  // Description:
  // If this flag is on (the default) then all time steps will be written out.
  // Otherwise, only the time already set in the pipeline will be used.
  vtkGetMacro(WriteAllTimeSteps, int);
  vtkSetMacro(WriteAllTimeSteps, int);
  vtkBooleanMacro(WriteAllTimeSteps, int);

  // Description:
  // As files are written, they will be tagged with an index indicating the time
  // step (starting with index 0).  This flag gives the number of characters to
  // use in the filename for the index (using 0 for padding).  Default is 4.
  vtkGetMacro(FileIndexSize, int);
  vtkSetClampMacro(FileIndexSize, int, 1, 6);

  virtual int ProcessRequest(vtkInformation *,
                             vtkInformationVector **,
                             vtkInformationVector *);

protected:
  vtkFileSeriesWriter();
  ~vtkFileSeriesWriter();

  char *FileName;

  vtkWriter *Writer;

  char *FileNameMethod;
  int WriteAllTimeSteps;
  int FileIndexSize;

  virtual void SetWriterFileName(const char *fname);

  virtual void WriteData() {}

  virtual int RequestInformation(vtkInformation *,
                                 vtkInformationVector **,
                                 vtkInformationVector *);
  virtual int RequestUpdateExtent(vtkInformation *,
                                  vtkInformationVector **,
                                  vtkInformationVector *);
  virtual int RequestData(vtkInformation *,
                          vtkInformationVector **,
                          vtkInformationVector *);

  virtual int CallWriter(vtkInformation *,
                         vtkInformationVector **,
                         vtkInformationVector *);

  int NumberOfTimeSteps;
  int CurrentTimeIndex;

private:
  vtkFileSeriesWriter(const vtkFileSeriesWriter &);     // Not implemented
  void operator=(const vtkFileSeriesWriter &);          // Not implemented
};

#endif //__vtkFileSeriesWriter_h
